#include <cstdio>
#include <omp.h>
#include <cmath>
#include "rngutil.h"
#include "worker.h"

int main(int argv, char* argc[]){
    const size_t n=1<<8;
    const size_t m=1<<12;
    const int nTrials=10;
    const int skipTrials=2;

    Charge_Dist chg;
    float* potential = (float*) malloc(sizeof(float)*n*n*n);

    // Initializing array of charges
    RNGutil rng[omp_get_max_threads()];
    printf("Initialization...");
    chg.x = (float*) malloc(sizeof(float)*m);
    chg.y = (float*) malloc(sizeof(float)*m);
    chg.z = (float*) malloc(sizeof(float)*m);
    chg.q = (float*) malloc(sizeof(float)*m);

#pragma omp parallel for schedule(guided)
    for (size_t i=0; i<m; i++) {
      const int iThread = omp_get_thread_num();
      chg.x[i] = rng[iThread].Next();
      chg.y[i] = rng[iThread].Next();
      chg.z[i] = rng[iThread].Next();
      chg.q[i] = rng[iThread].Next();
    }
    printf(" complete.\n");

    printf("\033[1m%5s %10s %8s\033[0m\n", "Trial", "Time, s", "GFLOP/s");
    double perf=0.0, dperf=0.0;
    for (int t=1; t<=nTrials; t++){
        potential[0:n*n]=0.0f;
        const double t0 = omp_get_wtime();
#pragma omp parallel for collapse(3) schedule(guided)
        for (int i = 0; i < n; i++) 
	  for (int j = 0; j < n; j++) 
	    for (int k = 0; k < n; k++) { 
	      CalculateElectricPotential(m, chg, i, j, k, potential[i*n*n+j*n+k]);
	    }
        const double t1 = omp_get_wtime();

	const double HztoPerf = 10.0*1e-9*double(n*n*n)*double(m);
	if (t > skipTrials) {
	  perf += HztoPerf/(t1-t0);
	  dperf += HztoPerf*HztoPerf/((t1-t0)*(t1-t0));
	}

	printf("%5d %10.3e %8.1f %s\n", 
	       t, (t1-t0), HztoPerf/(t1-t0), (t<=skipTrials?"*":""));
	fflush(stdout);
    }
    perf/=(double)(nTrials-skipTrials); 
    dperf=sqrt(dperf/(double)(nTrials-skipTrials)-perf*perf);
    printf("-----------------------------------------------------\n");
    printf("\033[1m%s %4s \033[42m%10.1f +- %.1f GFLOP/s\033[0m\n",
	   "Average performance:", "", perf, dperf);
    printf("-----------------------------------------------------\n");
    printf("* - warm-up, not included in average\n\n");
    free(potential);
}
