#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <omp.h>
#include <libzbc/zbc.h>


int main(int argc, char* argv[]) {

  if(argc < 4) {
    printf("ERROR: path size [zones]\n");
    exit(1);
  }

  struct zbc_device_info info;
  struct zbc_device *dev = NULL;
  struct zbc_zone *zones = NULL;
  struct zbc_zone *iozone = NULL;
  unsigned int nr_zones;
  char *path, *file = NULL;
  unsigned int ret;

  path = argv[1];
  
  /* Open device */
  ret = zbc_open(path, O_RDONLY, &dev);
  if ( ret != 0 ) {
    exit(1);
  }


#ifdef FLUSH
  int fd = open(path, O_RDONLY);
#endif


  /* Getting device info */
  ret = zbc_get_device_info(dev, &info);
  if ( ret < 0 ) {
    fprintf(stderr,
	    "zbc_get_device_info failed\n");
    exit(1);
  }
  

  /* Get zone list */
  ret = zbc_list_zones(dev, 0, ZBC_RO_ALL, &zones, &nr_zones);
  if ( ret != 0 ) {
    fprintf(stderr, "zbc_list_zones failed\n");
    ret = 1;
    exit(1);
  }

  /* Populating zone_ids[] (the list of the zones to work with) */
  const int nZones = argc-3;  
  int zone_ids[nZones]; 
  int z;
  for(z = 0; z < nZones; z++) {
    int zone_id = atoi(argv[z+3]);
    if(zone_id > 37255 || zone_id < 378) {
      fprintf(stderr, "Invalid zone: %d   Please choose a sequential zone (id = 378 - 37255)\n", zone_id);
      exit(2);
    }
    zone_ids[z] = zone_id;
  }

  /* Checking that the size is correct */
  const int size = atoi(argv[2]);
  const int log_block_size = info.zbd_logical_block_size;
  const int zone_length=(int)zbc_zone_length(&zones[zone_ids[0]]);
  const int max_size = 5505024; // set by the OS (so this value may be different)    
  if (size > max_size || size < log_block_size) {
    fprintf(stderr, "Invalid size %d: size must be in range (%d < size < %d)\n", size, log_block_size, max_size);
    exit(3);
  }
  if (zone_length*log_block_size % size != 0 || size % log_block_size != 0) {
    fprintf(stderr, "Invalid size %d: size must be a multiple of %d and divide out %d evenly\n", size, log_block_size, zone_length*log_block_size);
    exit(3);
  }


  /* Allocating the buffer */
  double* data_buffer;
  posix_memalign((void **) &data_buffer, 4096, max_size);


  /* Converting write size from Bytes to LBA count */
  const int lba_count = size / log_block_size;
  const int read_expected = zone_length;

  /* Starting the Benchmark Process */
  printf("Zone\tSize\tReads (LBA)\tTime (s)\tBW (MB/s)\n");
  int i;
  for(i=0; i<nZones; i++) { 
    int read=0;

    /* if FLUSH is defined, flushing the device caches */
#ifdef FLUSH
    zbc_flush(dev);
    fsync(fd);
#endif

    /* Reading the whole zone */
    int lba_offset;
    const double t0 = omp_get_wtime();
    for(lba_offset=0; lba_offset<zone_length; lba_offset+=lba_count) {
      read += zbc_pread(dev, &zones[zone_ids[i]],(void*) data_buffer, lba_count, lba_offset);
      
    }
    const double t_tot = omp_get_wtime() - t0;
    printf("%d\t%d\t%d\t\t%f\t%f\t%s\n", zone_ids[i], size, read, t_tot, (double) read* (double) log_block_size*1.0e-6/t_tot, (read==read_expected) ? "" : "* Some reads seem to have failed");
  }

  /* Cleaning up */
  free(data_buffer);
  zbc_close(dev);  
}
