#include <stdio.h>
#include <stdlib.h>

#include <mm_malloc.h>
#include <libzbc/zbc.h>

void PopulateData(void* write_buffer, int data_size); 

void VerifyData(void* read_buffer, int data_size); 

int main(int argc, char* argv[]) {
  struct zbc_device_info info;
  struct zbc_device *dev = NULL;
  struct zbc_zone *zones = NULL;
  unsigned int nr_zones;

  /* Opening and getting Information (see Section 3.1) */
  char *path = argv[1];
  unsigned int ret; 
  ret = zbc_open(path, O_RDWR, &dev);
  ret = zbc_get_device_info(dev, &info);

  /* Allocating I/O buffers, and populating the data to be written. (see Section 3.3.1) */
  const int data_size = 1<<20; // writing 1MB then reading 1MB
  void* write_buffer = (void*) _mm_malloc(data_size, 4096);
  void* read_buffer = (void*) _mm_malloc(data_size, 4096);
  PopulateData(write_buffer, data_size);  // Populate the data to be written

  /* Listing Zones and checking if there is enough space (see Section 3.2) */
  zbc_list_zones(dev, 0, ZBC_RO_ALL, &zones, &nr_zones);
  const int zone_id = atoi(argv[2]);
  const int rem_space_lba = zbc_zone_next_lba(&zones[zone_id])-zbc_zone_wp_lba(&zones[zone_id]);
  const int rem_space_bytes = rem_space_lba * info.zbd_logical_block_size;
  if(rem_space_bytes < data_size || zbc_zone_full(&zones[zone_id])) 
    { fprintf(stderr, "Not enough space in zone %d. Exiting\n", zone_id); exit(1); }
  printf("Space remaining in zone %d; %dB\n", zone_id, rem_space_bytes); 
    
  /* Writing the data (see Section 3.3.2) */
  int written;
  const int lba_count = data_size / info.zbd_logical_block_size;
  written = zbc_write(dev, &zones[zone_id],(void*) write_buffer, lba_count);
  if(written != lba_count) 
    { fprintf(stderr, "Write Failed. Written = %d. Exiting\n", written); exit(2); }

  /* Re-reading the data (see Section 3.3.2) */
  int read;
  const int lba_offset = zbc_zone_wp_lba(&zones[zone_id])-zbc_zone_start_lba(&zones[zone_id])-lba_count;
  read = zbc_pread(dev, &zones[zone_id],(void*) read_buffer, lba_count, lba_offset);
  if(read != lba_count) 
    { fprintf(stderr, "Read Failed. Read = %d. Exiting\n", read); exit(3); }

  VerifyData(read_buffer, data_size);

  /* Closing the drive (see 3.1) */
  ret = zbc_close(dev);
}


void PopulateData(void* write_buffer, int data_size) {
  const int doublesize = data_size/sizeof(double);
  double* data = (double*) write_buffer;
  int i;
  for(i = 0; i < doublesize; i++) {
    data[i] = (double) i;
  }
}

void VerifyData(void* read_buffer, int data_size) {
  const int doublesize = data_size/sizeof(double);
  double* data = (double*) read_buffer;
  double err = 0.0;
  int i;
  for(i = 0; i < doublesize; i++) {
    err+= (data[i] - (double) i) * (data[i] - (double) i);
  }
  if(err > 1.0e-6) {
    fprintf(stderr, "ERROR: Result does not match. err = %f\n", err);
  }
}

